package org.example.tools;

import java.time.LocalDateTime;
import java.util.Map;

import org.apache.lucene.search.IndexSearcher;

import dev.langchain4j.data.document.Metadata;
import dev.langchain4j.data.segment.TextSegment;
import dev.langchain4j.model.embedding.EmbeddingModel;
import dev.langchain4j.store.embedding.EmbeddingStore;
import dev.langchain4j.store.embedding.EmbeddingStoreIngestor;

public class VectorStoreService {
    private EmbeddingStore<TextSegment> embeddingStore;
    private final EmbeddingModel embeddingModel;
    private EmbeddingStoreIngestor ingestor;
    private final String collectionName;

    public VectorStoreService(EmbeddingModel embeddingModel, String collectionName) {
        this.embeddingModel = embeddingModel;
        this.collectionName = collectionName;

        // 使用Lucene作为向量存储
        try {
            // 向量维度为768
            this.embeddingStore = new VectorStore("lucene_vector_index", 768);
        } catch (Exception e) {
            throw new RuntimeException("Failed to initialize Lucene vector store", e);
        }

        this.ingestor = EmbeddingStoreIngestor.builder()
                .embeddingStore(embeddingStore)
                .embeddingModel(embeddingModel)
                .build();
    }

    // 添加文本到向量存储，并返回唯一ID
    public String addTextToVectorStore(String text, String source, LocalDateTime timestamp) {
        // 创建Metadata对象
        Map<String, String> metadataMap = Map.of(
                "source", source,
                "timestamp", timestamp.toString()
        );

        // 创建Metadata对象
        Metadata metadata = Metadata.from(metadataMap);
        TextSegment segment = TextSegment.from(text, metadata);

        // 向量化并存储
        String id = embeddingStore.add(embeddingModel.embed(text).content(), segment);

        System.out.println(String.format(TextStore.INFO_VECTORIZED_CONTENT,
                text.substring(0, Math.min(50, text.length()))));
        return id;
    }

    // 根据ID从向量存储中删除项目
    public void removeFromVectorStore(String id) {
        embeddingStore.remove(id);
        System.out.println(String.format(TextStore.INFO_REMOVED_FROM_VECTOR, id));
    }

    // 获取向量存储
    public EmbeddingStore<TextSegment> getEmbeddingStore() {
        return embeddingStore;
    }

    // 清空向量存储
    public void clearVectorStore() {
        try {
            ((VectorStore) embeddingStore).clear();
            System.out.println(TextStore.INFO_VECTOR_CLEARED);
        } catch (Exception e) {
            System.err.println(TextStore.ERROR_CLEARING_VECTOR + e.getMessage());
        }
    }

    // 获取搜索器
    public IndexSearcher getSearcher() {
        try {
            return ((VectorStore) embeddingStore).getSearcher();
        } catch (Exception e) {
            throw new RuntimeException("Failed to get searcher from vector store", e);
        }
    }
}