package org.example.animation;

import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import javax.imageio.ImageIO;

import org.example.tools.TextStore;

import com.fasterxml.jackson.databind.JsonNode;

/**
 * 精灵表加载器 - 用于解析Aseprite导出的JSON配置文件并加载精灵表动画
 */
public class SpriteSheetLoader {

    /**
     * 从资源路径加载指定情绪的精灵表动画
     * @param emotion 情绪名称（如"平常"、"开心"等）
     * @return 帧列表
     */
    public static List<Image> loadAnimationFromResource(String emotion) {
        List<Image> frames = new ArrayList<>();
        try {
            // 获取对应的资源路径
            String animDir = getAnimationDirByEmotion(emotion);
            if (animDir == null) {
                System.err.println("无法找到情绪对应的动画目录: " + emotion);
                return frames;
            }

            // 构建JSON和精灵表图像的路径
            String jsonPath = animDir + getJsonFileNameByEmotion(emotion);
            String imagePath = animDir + getImageFileNameByEmotion(emotion);

            // 读取JSON配置文件
            JsonNode jsonConfig = readJsonFromResource(jsonPath);
            if (jsonConfig == null) {
                return frames;
            }

            // 读取精灵表图像
            BufferedImage spriteSheet = readImageFromResource(imagePath);
            if (spriteSheet == null) {
                return frames;
            }

            // 根据JSON配置切割精灵表
            frames = extractFramesFromSpriteSheet(jsonConfig, spriteSheet);

        } catch (Exception e) {
            e.printStackTrace();
        }
        return frames;
    }

    /**
     * 根据情绪获取对应的动画目录
     */
    private static String getAnimationDirByEmotion(String emotion) {
        switch (emotion) {
            case "平常":
                return TextStore.NORMAL_ANIM_DIR;
            case "开心":
                return TextStore.HAPPY_ANIM_DIR;
            case "生气":
                return TextStore.ANGRY_ANIM_DIR;
            case "疑惑":
                return TextStore.CONFUSED_ANIM_DIR;
            case "难过":
                return TextStore.SAD_ANIM_DIR;
            default:
                return TextStore.NORMAL_ANIM_DIR;
        }
    }

    /**
     * 根据情绪获取对应的JSON文件名
     */
    private static String getJsonFileNameByEmotion(String emotion) {
        switch (emotion) {
            case "平常":
                return "normal.json";
            case "开心":
                return "happy.json";
            case "生气":
                return "angry.json";
            case "疑惑":
                return "confused.json";
            case "难过":
                return "sad.json";
            default:
                return "normal.json";
        }
    }

    /**
     * 根据情绪获取对应的精灵表图像文件名
     */
    private static String getImageFileNameByEmotion(String emotion) {
        switch (emotion) {
            case "平常":
                return "normal-sheet.png";
            case "开心":
                return "happy-sheet.png";
            case "生气":
                return "angry-sheet.png";
            case "疑惑":
                return "confused-sheet.png";
            case "难过":
                return "sad-sheet.png";
            default:
                return "normal-sheet.png";
        }
    }

    /**
     * 从资源路径读取JSON文件
     */
    private static JsonNode readJsonFromResource(String resourcePath) {
        try (InputStream inputStream = SpriteSheetLoader.class.getResourceAsStream(resourcePath)) {
            if (inputStream == null) {
                System.err.println("无法找到资源文件: " + resourcePath);
                return null;
            }
            return TextStore.OBJECT_MAPPER.readTree(inputStream);
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 从资源路径读取图像文件
     */
    private static BufferedImage readImageFromResource(String resourcePath) {
        try (InputStream inputStream = SpriteSheetLoader.class.getResourceAsStream(resourcePath)) {
            if (inputStream == null) {
                System.err.println("无法找到资源文件: " + resourcePath);
                return null;
            }
            return ImageIO.read(inputStream);
        } catch (IOException e) {
            e.printStackTrace();
            return null;
        }
    }

    /**
     * 根据JSON配置从精灵表中提取帧
     */
    private static List<Image> extractFramesFromSpriteSheet(JsonNode jsonConfig, BufferedImage spriteSheet) {
        List<Image> frames = new ArrayList<>();

        try {
            // 获取frames节点
            JsonNode framesNode = jsonConfig.get("frames");
            if (framesNode == null || !framesNode.isObject()) {
                return frames;
            }

            // 获取帧的名称并按顺序排序
            List<String> frameNames = new ArrayList<>();
            framesNode.fieldNames().forEachRemaining(frameNames::add);
            // 排序帧名称，确保动画顺序正确
            frameNames.sort(Comparator.naturalOrder());

            // 提取每一帧
            for (String frameName : frameNames) {
                JsonNode frameNode = framesNode.get(frameName);
                JsonNode frameRectNode = frameNode.get("frame");

                if (frameRectNode != null) {
                    int x = frameRectNode.get("x").asInt();
                    int y = frameRectNode.get("y").asInt();
                    int width = frameRectNode.get("w").asInt();
                    int height = frameRectNode.get("h").asInt();

                    // 确保坐标和尺寸有效
                    if (x >= 0 && y >= 0 && width > 0 && height > 0 && 
                            x + width <= spriteSheet.getWidth() && 
                            y + height <= spriteSheet.getHeight()) {
                        // 从精灵表中切割出单个帧
                        BufferedImage frame = spriteSheet.getSubimage(x, y, width, height);
                        frames.add(frame);
                    }
                }
            }

        } catch (Exception e) {
            e.printStackTrace();
        }

        return frames;
    }

    /**
     * 为Behave类加载所有情绪的精灵表动画
     * @param behave Behave实例
     */
    public static void loadAllEmotionsForBehave(Object behave) {
        if (behave == null) {
            return;
        }

        String[] emotions = {"平常", "开心", "生气", "疑惑", "难过"};
        for (String emotion : emotions) {
            List<Image> frames = loadAnimationFromResource(emotion);
            if (!frames.isEmpty()) {
                // 使用反射调用Behave的addAnimationFrames方法
                try {
                    java.lang.reflect.Method method = behave.getClass().getDeclaredMethod("addAnimationFrames", String.class, List.class);
                    method.setAccessible(true);
                    method.invoke(behave, emotion, frames);
                } catch (Exception e) {
                    e.printStackTrace();
                    System.err.println("无法为情绪" + emotion + "添加动画帧");
                }
            }
        }
    }
}