package org.example.ai;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;
import java.util.regex.Pattern;

import javax.swing.JOptionPane;

import org.apache.lucene.document.Document;
import org.apache.lucene.index.MultiTerms;
import org.apache.lucene.index.Terms;
import org.apache.lucene.index.TermsEnum;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.MatchAllDocsQuery;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.search.TotalHits;
import org.example.tools.TextStore;
import org.example.tools.VectorStoreService;

import dev.langchain4j.data.embedding.Embedding;
import dev.langchain4j.data.segment.TextSegment;
import dev.langchain4j.model.chat.ChatLanguageModel;
import dev.langchain4j.model.embedding.EmbeddingModel;
import dev.langchain4j.model.ollama.OllamaChatModel;
import dev.langchain4j.model.ollama.OllamaEmbeddingModel;
import dev.langchain4j.rag.content.retriever.EmbeddingStoreContentRetriever;
import dev.langchain4j.service.AiServices;
import dev.langchain4j.store.embedding.EmbeddingMatch;

public class Brain {
    private static List<String> conversationHistory = new ArrayList<>();
    private static String persona = TextStore.PERSONA;
    private static int currentTokenCount = 0;

    // 向量存储服务
    private static VectorStoreService vectorStoreService;
    private static EmbeddingModel embeddingModel;

    // 存储向量ID列表，用于快速统计
    private static List<String> vectorStoreIds = new ArrayList<>();

    // 使用统一的时间格式
    private static final DateTimeFormatter TIME_FORMATTER = DateTimeFormatter.ofPattern("yyyy年MM月dd日 HH时mm分ss秒");

    // 加载配置
    private static void loadConfiguration() {
        try {
            Path configPath = Paths.get(TextStore.CONFIG_FILE);
            
            if (Files.exists(configPath)) {
                try (InputStream is = new FileInputStream(configPath.toFile())) {
                    configProperties.load(is);
                    
                    // 更新默认值
                    if (configProperties.containsKey("model.name")) {
                        DEFAULT_MODEL_NAME = configProperties.getProperty("model.name");
                    }
                    if (configProperties.containsKey("model.baseUrl")) {
                        DEFAULT_BASE_URL = configProperties.getProperty("model.baseUrl");
                    }
                    if (configProperties.containsKey("knowledge.min_length")) {
                        try {
                            DEFAULT_KNOWLEDGE_MIN_LENGTH = Integer.parseInt(configProperties.getProperty("knowledge.min_length"));
                        } catch (NumberFormatException e) {
                            System.err.println("配置文件中knowledge.min_length格式错误，使用默认值: " + DEFAULT_KNOWLEDGE_MIN_LENGTH);
                        }
                    }
                    
                    System.out.println(TextStore.SUCCESS_CONFIG_LOADED);
                }
            } else {
                // 创建默认配置文件
                Files.createDirectories(configPath.getParent());
                try (OutputStream os = new FileOutputStream(configPath.toFile())) {
                    configProperties.setProperty("knowledge.min_length", "10");
                    configProperties.setProperty("model.name", "qwen2:7b");
                    configProperties.setProperty("model.baseUrl", "http://localhost:11434");
                    configProperties.setProperty("knowledge.similarity_threshold", "0.9");
                    configProperties.setProperty("knowledge.quality_min_length", "20");
                    configProperties.store(os, "Memosy Configuration");
                    
                    System.out.println(TextStore.SUCCESS_CONFIG_CREATED);
                }
            }
        } catch (IOException e) {
            System.err.println(TextStore.ERROR_LOADING_CONFIG + e.getMessage());
        }
    }

    // 简单问候模式
    private static final Pattern SIMPLE_GREETING = Pattern.compile(
            "(?i)(你好|嗨|早上好|下午好|晚上好|晚安|拜拜|再见|谢谢|不客气)");

    public static void main(String[] args) {
        // 初始化嵌入模型
        embeddingModel = OllamaEmbeddingModel.builder()
                .baseUrl("http://localhost:11434")
                .modelName("nomic-embed-text")
                .build();

        vectorStoreService = new VectorStoreService(embeddingModel, TextStore.DATA_DIR + "ai_knowledge_base");

        // 加载向量ID列表（用于快速统计）
        loadVectorStoreIds();

        // 加载历史记录和配置
        loadConversationHistory();
        loadConfiguration();

        // 添加程序启动消息
        String startupTime = LocalDateTime.now().format(TIME_FORMATTER);
        String startupMessage = startupTime + " - " + TextStore.SYSTEM_STARTUP;
        conversationHistory.add(startupMessage);
        updateTokenCount(startupMessage, true);
        saveConversationHistory();

        // 添加关闭钩子
        Runtime.getRuntime().addShutdownHook(new Thread(() -> {
            String shutdownTime = LocalDateTime.now().format(TIME_FORMATTER);
            String shutdownMessage = shutdownTime + " - " + TextStore.SYSTEM_SHUTDOWN;
            conversationHistory.add(shutdownMessage);
            updateTokenCount(shutdownMessage, true);
            saveConversationHistory();
            saveVectorStoreIds();

            // 维护知识库
            maintainKnowledgeBase();
        }));

        // 启动GUI界面
        javax.swing.SwingUtilities.invokeLater(() -> {
            Appearance appearance = new Appearance();
            appearance.setBrain(new Brain());
            appearance.createAndShowGUI();
        });
    }


    // 加载向量ID列表
    private static void loadVectorStoreIds() {
        Path path = Paths.get(TextStore.VECTOR_IDS_FILE);
        if (Files.exists(path)) {
            try (ObjectInputStream ois = new ObjectInputStream(new FileInputStream(TextStore.VECTOR_IDS_FILE))) {
                vectorStoreIds = (List<String>) ois.readObject();
                System.out.printf((TextStore.SUCCESS_VECTOR_IDS_LOADED) + "%n", vectorStoreIds.size());
            } catch (IOException | ClassNotFoundException e) {
                System.err.println(TextStore.ERROR_LOADING_VECTOR_IDS + e.getMessage());
                vectorStoreIds = new ArrayList<>();
            }
        } else {
            System.out.println(TextStore.INFO_NO_VECTOR_IDS_FILE);
            vectorStoreIds = new ArrayList<>();
        }
    }

    // 保存向量ID列表
    private static void saveVectorStoreIds() {
        try (ObjectOutputStream oos = new ObjectOutputStream(new FileOutputStream(TextStore.VECTOR_IDS_FILE))) {
            oos.writeObject(vectorStoreIds);
            System.out.println(TextStore.SUCCESS_VECTOR_IDS_SAVED);
        } catch (IOException e) {
            System.err.println(TextStore.ERROR_SAVING_VECTOR_IDS + e.getMessage());
        }
    }

    // 添加知识到向量存储
    public static void addToKnowledgeBase(String content, String source) {
        // 向量化内容
        String vectorId = vectorStoreService.addTextToVectorStore(content, source, LocalDateTime.now());
        vectorStoreIds.add(vectorId);

        // 限制知识库大小
        if (vectorStoreIds.size() > 4000) {
            // 从向量存储中删除最旧的项目
            removeOldestVectorItem();
            vectorStoreIds.remove(0);
        }

        saveVectorStoreIds();
        System.out.println(String.format(TextStore.SUCCESS_KNOWLEDGE_ADDED, content));
    }

    // 从向量存储中删除最旧的项目
    private static void removeOldestVectorItem() {
        try {
            // 查询最旧的项目
            IndexSearcher searcher = vectorStoreService.getSearcher();
            Sort sort = new Sort(new SortField("timestamp", SortField.Type.LONG, true));
            TopDocs topDocs = searcher.search(new MatchAllDocsQuery(), 1, sort);
            
            if (topDocs.scoreDocs.length > 0) {
                Document doc = searcher.doc(topDocs.scoreDocs[0].doc);
                String id = doc.get("id");
                vectorStoreService.removeFromVectorStore(id);
            }
        } catch (Exception e) {
            System.err.println("删除最旧项目失败: " + e.getMessage());
        }
    }

    // 清空向量存储
    public void clearVectorStore() {
        vectorStoreService.clearVectorStore();
        vectorStoreIds.clear();
        saveVectorStoreIds();
        JOptionPane.showMessageDialog(null, "向量知识库已清空");
    }
    
    // 基于上下文分析当前应使用的情绪 - 简化版，避免关键词分析带来的死板感
    private static String analyzeEmotionFromContext(String userInput, String history) {
        try {
            // 默认使用平常情绪，让AI回复更加自然和灵活
            return "平常";
        } catch (Exception e) {
            System.err.println("情绪分析出错: " + e.getMessage());
            return "平常"; // 出错时默认使用平常情绪
        }
    }

    // 从文件导入知识库
    public void importKnowledgeFromFile(File file) {
        try {
            List<String> lines = Files.readAllLines(file.toPath());
            int count = 0;
            for (String line : lines) {
                if (!line.trim().isEmpty()) {
                    addToKnowledgeBase(line.trim(), "import");
                    count++;
                }
            }
            JOptionPane.showMessageDialog(null,
                    String.format(TextStore.SUCCESS_IMPORTED_KNOWLEDGE, count));
        } catch (IOException e) {
            JOptionPane.showMessageDialog(null,
                    TextStore.ERROR_IMPORTING_FILE + e.getMessage(), "错误", JOptionPane.ERROR_MESSAGE);
        }
    }
    
    // 配置属性
    private static Properties configProperties = new Properties();
    private static String DEFAULT_MODEL_NAME = "qwen2:7b";
    private static String DEFAULT_BASE_URL = "http://localhost:11434";
    private static int DEFAULT_KNOWLEDGE_MIN_LENGTH = 10;
    
    // 获取AI回复使用RAG
    public String getAIResponse(String userInput) {
        // 记录用户输入
        String timestamp = LocalDateTime.now().format(TIME_FORMATTER);
        String userMessage = timestamp + " - 用户: " + userInput;
        conversationHistory.add(userMessage);

        // 更新token计数
        updateTokenCount(userMessage, true);

        // 检查并修剪历史记录
        trimHistoryIfNeeded();

        try {
            // 从配置中获取模型信息
            String modelName = DEFAULT_MODEL_NAME;
            String baseUrl = DEFAULT_BASE_URL;
            
            if (configProperties.containsKey("model.name")) {
                modelName = configProperties.getProperty("model.name");
            }
            if (configProperties.containsKey("model.baseUrl")) {
                baseUrl = configProperties.getProperty("model.baseUrl");
            }
            
            // 创建Ollama聊天模型
            ChatLanguageModel model = OllamaChatModel.builder()
                    .baseUrl(baseUrl)
                    .modelName(modelName)
                    .timeout(java.time.Duration.ofSeconds(300))
                    .build();

            // 创建内容检索器，从向量存储中获取相关知识
            EmbeddingStoreContentRetriever contentRetriever = EmbeddingStoreContentRetriever.builder()
                    .embeddingStore(vectorStoreService.getEmbeddingStore())
                    .embeddingModel(embeddingModel)
                    .maxResults(2) // 减少检索结果数量，避免过多重复
                    .build();

            // 创建AI服务，使用RAG
            Assistant assistant = AiServices.builder(Assistant.class)
                    .chatLanguageModel(model)
                    .contentRetriever(contentRetriever)
                    .build();

            // 组合提示词 - 只使用最近的历史消息，去掉时间戳使格式更清晰
            StringBuilder historyBuilder = new StringBuilder();
            int startIndex = Math.max(0, conversationHistory.size() - TextStore.MAX_HISTORY_ITEMS);
            for (int i = startIndex; i < conversationHistory.size(); i++) {
                String message = conversationHistory.get(i);
                // 去掉时间戳，只保留角色和内容，使格式更清晰
                if (message.contains(" - 用户: ")) {
                    message = "用户: " + message.substring(message.indexOf(" - 用户: ") + 7);
                } else if (message.contains(" - AI: ")) {
                    message = "AI: " + message.substring(message.indexOf(" - AI: ") + 6);
                }
                historyBuilder.append(message).append("\n");
            }
            
            // 分析当前应使用的情绪
            String analyzedEmotion = analyzeEmotionFromContext(userInput, historyBuilder.toString());
            System.out.println("分析到的情绪: " + analyzedEmotion);

            // 使用TextStore中的模板
            String prompt = String.format(TextStore.PROMPT_TEMPLATE,
                    persona, "平常", "平常", historyBuilder.toString());

            // 获取模型回复
            String rawResponse = assistant.chat(prompt);
            String response = enforceResponseFormat(extractPureResponse(rawResponse));
            
            // 检查AI回复是否与用户输入重复
            String contentWithoutEmotion = response.substring(response.indexOf("]") + 1).trim();
            if (calculateSimilarity(contentWithoutEmotion, userInput) > 0.7) {
                // 如果回复与用户输入高度相似，生成一个新的回复
                // 从响应中提取情绪标签
                String emotion = "平常";
                java.util.regex.Pattern pattern = java.util.regex.Pattern.compile("^\\[(.*?)\\]\\s*");
                java.util.regex.Matcher matcher = pattern.matcher(response);
                if (matcher.find()) {
                    emotion = matcher.group(1).trim();
                }
                response = generateMoreVariedResponse(emotion);
                System.out.println("检测到回复与用户输入重复，已生成新回复");
            }

            // 记录AI回复
            String aiTimestamp = LocalDateTime.now().format(TIME_FORMATTER);
            String aiMessage = aiTimestamp + " - AI: " + response;
            conversationHistory.add(aiMessage);

            // 智能判断是否添加到知识库
            if (isWorthAddingToKnowledge(userInput, response)) {
                // 添加用户问题
                addToKnowledgeBase(userInput, "user_question");

                // 添加AI的高质量回答
                addToKnowledgeBase(response, "ai_knowledge");

                System.out.println("已添加问答到知识库");
            }

            // 更新token计数
            updateTokenCount(aiMessage, true);

            // 检查并修剪历史记录
            trimHistoryIfNeeded();

            // 保存历史记录
            saveConversationHistory();

            return response;

        } catch (Exception e) {
            String errorMsg = "发生错误: " + e.getMessage();
            String errorTimestamp = LocalDateTime.now().format(TIME_FORMATTER);
            String systemMessage = errorTimestamp + " - 系统: " + errorMsg;
            conversationHistory.add(systemMessage);

            // 更新token计数
            updateTokenCount(systemMessage, true);

            // 检查并修剪历史记录
            trimHistoryIfNeeded();

            saveConversationHistory();
            return errorMsg + "\n请确保Ollama服务器正在运行，且模型名称正确。";
        }
    }

    // 强制响应格式和长度
    private String enforceResponseFormat(String response) {
        // 修复正则表达式，正确匹配 [情绪] 格式
        java.util.regex.Pattern pattern = java.util.regex.Pattern.compile("^\\[(.*?)\\]\\s*");
        java.util.regex.Matcher matcher = pattern.matcher(response.trim());
        boolean hasEmotionFormat = matcher.find();
        String emotion = "平常";
        String content = response.trim();

        if (hasEmotionFormat) {
            // 提取情绪和内容
            emotion = matcher.group(1).trim();
            content = response.substring(matcher.end()).trim();
            
            // 验证情绪是否有效
            if (!emotion.equals("生气") && !emotion.equals("难过") && 
                !emotion.equals("开心") && !emotion.equals("疑惑") && 
                !emotion.equals("平常")) {
                emotion = "平常"; // 无效情绪时使用默认值
            }
        } else {
            // 如果没有情绪标签，使用默认情绪"平常"
            // 不再通过关键词推断情绪，避免误判
            emotion = "平常";
        }
        
        // 检查内容中是否包含其他情绪表达（如：生气: 难过: 等格式）
        java.util.regex.Pattern contentEmotionPattern = java.util.regex.Pattern.compile("^(生气|难过|开心|疑惑|平常):\\s*");
        java.util.regex.Matcher contentMatcher = contentEmotionPattern.matcher(content);
        if (contentMatcher.find()) {
            // 提取内容中的情绪
            String contentEmotion = contentMatcher.group(1).trim();
            // 优先使用内容中的情绪
            emotion = contentEmotion;
            // 移除内容中的情绪前缀
            content = content.substring(contentMatcher.end()).trim();
        }
        
        // 确保内容不为空
        if (content.isEmpty()) {
            content = "我现在感觉很好，有什么我可以帮助你的吗？";
        }
        
        // 重新构建格式正确的响应，确保只有一个情绪标签
        response = "[" + emotion + "] " + content;

        // 检查响应长度，强制限制在3-4句话
        String[] sentences = content.split("[。！？.!?]");
        if (sentences.length > 4) {
            // 注意：这里拆分和重组时，要保留开头的 [情绪] 标签
            StringBuilder shortenedContent = new StringBuilder();
            for (int i = 0; i < Math.min(4, sentences.length); i++) {
                if (!sentences[i].trim().isEmpty()) {
                    shortenedContent.append(sentences[i].trim());
                    if (i < Math.min(4, sentences.length) - 1) {
                        shortenedContent.append("。");
                    }
                }
            }
            // 重新组合情绪标签和缩短后的内容
            response = "[" + emotion + "] " + shortenedContent.toString();
        }
        
        // 确保响应内容多样化，避免重复
        if (isResponseTooShortOrRepetitive(response)) {
            response = generateMoreVariedResponse(emotion);
        }
        
        return response;
    }
    
    // 检查响应是否太短或重复性高
    private boolean isResponseTooShortOrRepetitive(String response) {
        // 检查长度
        if (response.length() < 10) {
            return true;
        }
        
        // 检查是否与最近的几条回复重复
        int recentCheckCount = 3;
        if (conversationHistory.size() >= recentCheckCount) {
            for (int i = 0; i < recentCheckCount; i++) {
                String recentResponse = conversationHistory.get(conversationHistory.size() - 1 - i);
                if (recentResponse.contains(" - AI: ")) {
                    recentResponse = recentResponse.substring(recentResponse.indexOf(" - AI: ") + 6);
                    // 提取内容部分进行比较（忽略情绪标签）
                    if (recentResponse.contains("]")) {
                        recentResponse = recentResponse.substring(recentResponse.indexOf("]") + 1).trim();
                        String currentContent = response.substring(response.indexOf("]") + 1).trim();
                        // 如果内容相似度高，认为重复
                        if (calculateSimilarity(recentResponse, currentContent) > 0.7) {
                            return true;
                        }
                    }
                }
            }
        }
        
        return false;
    }
    
    // 生成更多样化的响应
    private String generateMoreVariedResponse(String emotion) {
        // 根据情绪生成不同的备用回复
        Map<String, String[]> fallbackResponses = new HashMap<>();
        fallbackResponses.put("平常", new String[] {
            "今天天气真不错，你有什么计划吗？",
            "我在这里随时为你提供帮助。",
            "有什么有趣的事情想和我分享吗？",
            "我们来聊点什么吧！"
        });
        fallbackResponses.put("开心", new String[] {
            "哈哈，今天心情真好！",
            "能帮到你我很开心！",
            "和你聊天真有趣！",
            "你总是能让我感到快乐！"
        });
        fallbackResponses.put("难过", new String[] {
            "别难过，一切都会好起来的。",
            "我在这里陪着你。",
            "有什么不开心的事可以告诉我。",
            "希望我的陪伴能让你感觉好一点。"
        });
        fallbackResponses.put("生气", new String[] {
            "冷静一下，生气对身体不好哦。",
            "发生什么让你这么生气的事了？",
            "深呼吸，慢慢来。",
            "我理解你的感受，但请不要生气太久。"
        });
        fallbackResponses.put("疑惑", new String[] {
            "这个问题有点难，让我想想。",
            "我不太确定，我们可以一起研究。",
            "能再详细解释一下吗？",
            "我有点困惑，能给我更多信息吗？"
        });
        
        // 选择一个随机的备用回复
        String[] responses = fallbackResponses.getOrDefault(emotion, fallbackResponses.get("平常"));
        String randomResponse = responses[(int)(Math.random() * responses.length)];
        
        return "[" + emotion + "] " + randomResponse;
    }
    
    // 计算字符串相似度（简化版）
    private double calculateSimilarity(String s1, String s2) {
        if (s1.isEmpty() || s2.isEmpty()) {
            return 0;
        }
        
        // 使用Jaccard相似度算法的简化版本
        Set<String> set1 = new HashSet<>(Arrays.asList(s1.split(" ")));
        Set<String> set2 = new HashSet<>(Arrays.asList(s2.split(" ")));
        Set<String> intersection = new HashSet<>(set1);
        intersection.retainAll(set2);
        Set<String> union = new HashSet<>(set1);
        union.addAll(set2);
        
        return (double)intersection.size() / union.size();
    }

    // 知识质量评估器
    private boolean isWorthAddingToKnowledge(String userInput, String aiResponse) {
        // 排除过短内容
        if (userInput.length() < DEFAULT_KNOWLEDGE_MIN_LENGTH || aiResponse.length() < 20) {
            System.out.println(String.format(TextStore.INFO_KNOWLEDGE_SKIPPED_LOW_QUALITY,
                    "内容过短: " + userInput));
            return false;
        }

        // 排除简单问候和告别
        if (SIMPLE_GREETING.matcher(userInput).find()) {
            System.out.println(String.format(TextStore.INFO_KNOWLEDGE_SKIPPED_LOW_QUALITY,
                    "简单问候: " + userInput));
            return false;
        }

        // 排除重复性内容（基于语义相似度）
        if (isSimilarToExistingKnowledge(userInput)) {
            System.out.println(String.format(TextStore.INFO_KNOWLEDGE_SKIPPED_DUPLICATE, userInput));
            return false;
        }

        // 移除了知识关键词检查，让回答更自然

        // AI回答质量评估
        if (isHighQualityResponse(aiResponse)) {
            System.out.printf((TextStore.INFO_KNOWLEDGE_ADDED_QUALITY) + "%n",
                    "高质量回答: " + aiResponse.substring(0, Math.min(30, aiResponse.length())));
            return true;
        }

        System.out.printf((TextStore.INFO_KNOWLEDGE_SKIPPED_LOW_QUALITY) + "%n",
                "未通过质量检查: " + userInput);
        return false;
    }

    // 语义相似度检查
    private boolean isSimilarToExistingKnowledge(String text) {
        try {
            // 获取配置文件中的相似度阈值，如果没有或格式错误则使用默认值
            double similarityThreshold = 0.9;
            try {
                if (configProperties.containsKey("knowledge.similarity_threshold")) {
                    similarityThreshold = Double.parseDouble(configProperties.getProperty("knowledge.similarity_threshold"));
                }
            } catch (NumberFormatException e) {
                System.err.println("配置文件中knowledge.similarity_threshold格式错误，使用默认值: " + similarityThreshold);
            }
            
            // 使用嵌入模型计算相似度
            Embedding newEmbedding = embeddingModel.embed(text).content();
            List<EmbeddingMatch<TextSegment>> similarItems = 
                    vectorStoreService.getEmbeddingStore().findRelevant(newEmbedding, 3, similarityThreshold);

            return !similarItems.isEmpty();
        } catch (Exception e) {
            return false;
        }
    }



    // 回答质量评估
    private boolean isHighQualityResponse(String response) {
        // 检查回答长度适中
        if (response.length() < 20 || response.length() > 100) {
            return false;
        }

        // 检查是否包含具体信息（而非敷衍）
        return response.matches(".*[0-9A-Za-z\u4e00-\u9fa5]{10,}.*");
    }

    private String extractPureResponse(String rawResponse) {
        // 移除可能包含的时间戳和AI前缀
        if (rawResponse.contains(" - AI: ")) {
            String[] parts = rawResponse.split(" - AI: ");
            return parts[parts.length - 1];
        }
        return rawResponse;
    }

    private static void updateTokenCount(String message, boolean add) {
        int chineseChars = countChineseCharacters(message);
        int otherChars = message.length() - chineseChars;
        int messageTokens = (int)(chineseChars * 1.5 + otherChars * 0.25);

        if (add) {
            currentTokenCount += messageTokens;
        } else {
            currentTokenCount -= messageTokens;
        }
    }

    private static void trimHistoryIfNeeded() {
        while (conversationHistory.size() > TextStore.MAX_HISTORY_ITEMS || currentTokenCount > TextStore.MAX_TOKENS) {
            if (conversationHistory.isEmpty()) break;

            String removedMessage = conversationHistory.remove(0);
            updateTokenCount(removedMessage, false);
        }

        if (conversationHistory.size() < TextStore.MAX_HISTORY_ITEMS && currentTokenCount < TextStore.MAX_TOKENS) {
            return;
        }

        System.out.println(String.format(TextStore.INFO_HISTORY_TRIMMED,
                conversationHistory.size(), currentTokenCount));
    }

    public List<String> getConversationHistory() {
        return new ArrayList<>(conversationHistory);
    }

    public void clearConversationHistory() {
        conversationHistory.clear();
        currentTokenCount = 0;
        saveConversationHistory();
    }

    private static int countChineseCharacters(String text) {
        int count = 0;
        for (char c : text.toCharArray()) {
            if (c >= '\u4E00' && c <= '\u9FFF') {
                count++;
            }
        }
        return count;
    }

    private static void loadConversationHistory() {
        Path path = Paths.get(TextStore.HISTORY_FILE);
        if (Files.exists(path)) {
            try {
                // 检查文件是否为空或只有一个字符
                long fileSize = Files.size(path);
                if (fileSize <= 1) {
                    System.out.println(TextStore.INFO_NO_HISTORY_FILE);
                    conversationHistory = new ArrayList<>();
                    return;
                }
                
                // 添加UTF-8编码设置，解决中文乱码问题
                conversationHistory = Files.readAllLines(path, java.nio.charset.StandardCharsets.UTF_8);
                System.out.println(String.format(TextStore.SUCCESS_HISTORY_LOADED, conversationHistory.size()));
                estimateTokenCount();
                trimHistoryIfNeeded();
            } catch (IOException e) {
                // 更详细的错误处理，捕获不同类型的IO异常
                System.err.println(TextStore.ERROR_LOADING_HISTORY + e.getMessage());
                
                // 尝试创建一个新的空历史记录文件
                try {
                    Files.createDirectories(path.getParent());
                    Files.write(path, new ArrayList<>(), java.nio.charset.StandardCharsets.UTF_8);
                    System.out.println("已创建新的历史记录文件");
                } catch (IOException ex) {
                    System.err.println("创建新历史记录文件失败: " + ex.getMessage());
                }
                
                conversationHistory = new ArrayList<>();
            } catch (Exception e) {
                // 捕获其他可能的异常，如格式错误
                System.err.println("读取历史记录时发生未知错误: " + e.getMessage());
                conversationHistory = new ArrayList<>();
            }
        } else {
            try {
                // 确保父目录存在
                Files.createDirectories(path.getParent());
                // 创建新的历史记录文件
                Files.write(path, new ArrayList<>(), java.nio.charset.StandardCharsets.UTF_8);
                System.out.println(TextStore.INFO_NO_HISTORY_FILE);
            } catch (IOException e) {
                System.err.println("创建新历史记录文件失败: " + e.getMessage());
            }
            conversationHistory = new ArrayList<>();
        }
    }

    private static void estimateTokenCount() {
        currentTokenCount = 0;
        for (String message : conversationHistory) {
            int chineseChars = countChineseCharacters(message);
            int otherChars = message.length() - chineseChars;
            currentTokenCount += (int)(chineseChars * 1.5 + otherChars * 0.25);
        }
        System.out.println(String.format(TextStore.INFO_CURRENT_TOKENS, currentTokenCount));
    }

    public static void saveConversationHistory() {
        try {
            Path path = Paths.get(TextStore.HISTORY_FILE);
            // 确保父目录存在
            Files.createDirectories(path.getParent());
            // 使用UTF-8编码写入文件
            Files.write(path, conversationHistory, java.nio.charset.StandardCharsets.UTF_8);
            System.out.println(TextStore.SUCCESS_HISTORY_SAVED);
        } catch (IOException e) {
            System.err.println(TextStore.ERROR_SAVING_HISTORY + e.getMessage());
        }
    }

    // 知识库维护方法
    public static void maintainKnowledgeBase() {
        try {
            // 检查vectorStoreService是否可用
            if (vectorStoreService == null) {
                System.out.println("向量存储服务未初始化，跳过知识库维护");
                return;
            }
            
            System.out.println("开始知识库维护...");
            
            IndexSearcher searcher = null;
            try {
                searcher = vectorStoreService.getSearcher();
                TopDocs topDocs = searcher.search(new MatchAllDocsQuery(), Integer.MAX_VALUE);
                
                int removedCount = 0;
                for (int i = 0; i < topDocs.scoreDocs.length; i++) {
                    try {
                        Document doc = searcher.doc(topDocs.scoreDocs[i].doc);
                        String content = doc.get("text");
                        String source = doc.get("meta_source");
                        
                        // 检查是否为低质量内容
                        if (content != null && (content.length() < 15 || 
                            SIMPLE_GREETING.matcher(content).find() ||
                            (source != null && source.equals("simple_greeting")))) {
                            
                            String id = doc.get("id");
                            vectorStoreService.removeFromVectorStore(id);
                            vectorStoreIds.remove(id);
                            removedCount++;
                        }
                    } catch (Exception e) {
                        System.err.println("处理文档时出错: " + e.getMessage());
                        // 继续处理下一个文档
                    }
                }
                
                if (removedCount > 0) {
                    System.out.println("已移除 " + removedCount + " 条低质量知识");
                }
            } catch (Exception e) {
                System.err.println("搜索文档时出错: " + e.getMessage());
            }
            
            // 合并相似内容
            try {
                deduplicateSimilarKnowledge();
            } catch (Exception e) {
                System.err.println("去重过程中出错: " + e.getMessage());
            }
            
            // 保存更新后的向量ID列表
            try {
                saveVectorStoreIds();
            } catch (Exception e) {
                System.err.println("保存向量ID列表时出错: " + e.getMessage());
            }
            
            // 保存历史记录
            try {
                saveConversationHistory();
            } catch (Exception e) {
                System.err.println("保存对话历史时出错: " + e.getMessage());
            }
            
            System.out.println("知识库维护完成，当前条目: " + vectorStoreIds.size());
        } catch (Exception e) {
            System.err.println("知识库维护失败: " + e.getMessage());
            // 打印完整的异常堆栈，帮助调试
            e.printStackTrace();
        }
    }

    // 相似内容去重
    private static void deduplicateSimilarKnowledge() {
        try {
            IndexSearcher searcher = vectorStoreService.getSearcher();
            TopDocs topDocs = searcher.search(new MatchAllDocsQuery(), Integer.MAX_VALUE);
            
            List<String> toRemove = new ArrayList<>();
            
            for (int i = 0; i < topDocs.scoreDocs.length; i++) {
                Document doc1 = searcher.doc(topDocs.scoreDocs[i].doc);
                String content1 = doc1.get("text");
                String id1 = doc1.get("id");
                
                for (int j = i + 1; j < topDocs.scoreDocs.length; j++) {
                    Document doc2 = searcher.doc(topDocs.scoreDocs[j].doc);
                    String content2 = doc2.get("text");
                    String id2 = doc2.get("id");
                    
                    // 简单的内容相似度检查
                    if (Math.abs(content1.length() - content2.length()) < 10 &&
                        (content1.contains(content2.substring(0, Math.min(10, content2.length()))) ||
                        content2.contains(content1.substring(0, Math.min(10, content1.length()))))) {
                        
                        // 保留较新的内容，删除较旧的
                        long timestamp1 = Long.parseLong(doc1.get("timestamp"));
                        long timestamp2 = Long.parseLong(doc2.get("timestamp"));
                        
                        if (timestamp1 < timestamp2) {
                            toRemove.add(id1);
                        } else {
                            toRemove.add(id2);
                        }
                    }
                }
            }
            
            // 移除重复内容
            for (String id : toRemove) {
                vectorStoreService.removeFromVectorStore(id);
                vectorStoreIds.remove(id);
            }
            
            if (!toRemove.isEmpty()) {
                System.out.println("已移除 " + toRemove.size() + " 条重复知识");
            }
        } catch (Exception e) {
            System.err.println("去重失败: " + e.getMessage());
        }
    }

    // 知识库统计方法
    public void printKnowledgeStats() {
        System.out.println("=== 知识库统计 ===");
        
        try {
            IndexSearcher searcher = vectorStoreService.getSearcher();
            TotalHits totalHits = searcher.search(new MatchAllDocsQuery(), 1).totalHits;
            System.out.println("总条目: " + totalHits.value);
            
            // 按来源统计
            Terms terms = MultiTerms.getTerms(searcher.getIndexReader(), "meta_source");
            if (terms != null) {
                TermsEnum termsEnum = terms.iterator();
                while (termsEnum.next() != null) {
                    String source = termsEnum.term().utf8ToString();
                    long count = termsEnum.docFreq();
                    System.out.println(source + ": " + count);
                }
            }
            
            // 获取内容长度统计
            long totalChars = 0;
            int docCount = 0;
            
            TopDocs allDocs = searcher.search(new MatchAllDocsQuery(), Integer.MAX_VALUE);
            for (int i = 0; i < allDocs.scoreDocs.length; i++) {
                Document doc = searcher.doc(allDocs.scoreDocs[i].doc);
                String content = doc.get("text");
                if (content != null) {
                    totalChars += content.length();
                    docCount++;
                }
            }
            
            double avgChars = docCount > 0 ? (double) totalChars / docCount : 0;
            System.out.println("总字符数: " + totalChars);
            System.out.println("平均长度: " + String.format("%.1f", avgChars) + " 字符");
            
        } catch (Exception e) {
            System.err.println("获取统计信息失败: " + e.getMessage());
            System.out.println("使用向量ID列表进行统计:");
            System.out.println("总条目: " + vectorStoreIds.size());
        }
    }

    // 获取向量存储服务
    public VectorStoreService getVectorStoreService() {
        return vectorStoreService;
    }

    // 获取嵌入模型
    public EmbeddingModel getEmbeddingModel() {
        return embeddingModel;
    }
}