package org.example.ai;

import java.awt.AlphaComposite;
import java.awt.Color;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.imageio.ImageIO;
import javax.swing.JPanel;
import javax.swing.Timer;

import org.example.tools.TextStore;

public class Behave {
    // 情绪与对应的动画帧列表
    private Map<String, List<Image>> emotionFrames = new HashMap<>();
    // 当前播放的动画帧列表
    private List<Image> currentFrames;
    // 当前帧索引
    private int currentFrameIndex = 0;
    // 动画计时器
    private Timer animationTimer;
    // 帧率（毫秒/帧） - 修改为350ms，使动画播放更缓慢
    private int frameRate = 350;
    // 当前情绪
    private String currentEmotion = "平常";
    // 目标情绪（用于平滑过渡）
    private String targetEmotion = null;
    // 过渡计时器
    private int transitionCounter = 0;
    // 过渡所需帧数
    private int transitionFrames = 5;
    // 动画播放完成后的保持时间（毫秒）
    private int holdTime = 2000;
    // 动画播放完成后的保持计时器
    private Timer holdTimer;
    // 是否处于保持状态
    private boolean isHolding = false;
    // 是否可以切换情绪
    private boolean canSwitchEmotion = true;

    // 动画面板（用于刷新）
    private JPanel animationPanel;
    // 单例实例
    private static Behave instance;

    private Behave() {
        // 私有构造函数防止外部实例化
        loadAnimationFrames();
        initTimers();
    }

    // 获取单例实例
    public static Behave getInstance() {
        if (instance == null) {
            instance = new Behave();
        }
        return instance;
    }

    // 设置动画面板（用于刷新）
    public void setAnimationPanel(JPanel panel) {
        this.animationPanel = panel;
    }

    // 加载动画帧
    private void loadAnimationFrames() {
        // 定义情绪和对应的图像目录
        String[] emotions = {"平常", "开心", "难过", "生气", "疑惑"};
        String basePath = "src/main/resources/animations/";

        for (String emotion : emotions) {
            List<Image> frames = new ArrayList<>();
            try {
                // 加载该情绪的所有帧
                int frameIndex = 1;
                File frameFile;
                do {
                    frameFile = new File(basePath + emotion + "/" + frameIndex + ".png");
                    if (frameFile.exists()) {
                        BufferedImage image = ImageIO.read(frameFile);
                        // 先保持原始图像大小，绘制时再根据面板大小缩放
                        frames.add(image);
                        frameIndex++;
                    }
                } while (frameFile.exists());

                // 如果没有找到动画帧，使用默认图像
                if (frames.isEmpty()) {
                    frames.add(createDefaultImage(emotion));
                }

                emotionFrames.put(emotion, frames);
            } catch (IOException e) {
                e.printStackTrace();
                // 出错时使用默认图像
                frames.add(createDefaultImage(emotion));
                emotionFrames.put(emotion, frames);
            }
        }

        // 设置初始动画帧
        currentFrames = emotionFrames.get("平常");
    }

    // 绘制当前帧
    public void paintCurrentFrame(Graphics g, JPanel panel) {
        if (currentFrames != null && !currentFrames.isEmpty()) {
            Image currentFrame = currentFrames.get(currentFrameIndex);
            if (currentFrame != null) {
                // 获取面板的尺寸
                int panelWidth = panel.getWidth();
                int panelHeight = panel.getHeight();
                
                // 设置合适的图像宽高比（例如4:5）
                int imageWidth = TextStore.DEFAULT_SIZE * 2;  // 增加基础尺寸
                int imageHeight = (int)(imageWidth * 1.25);  // 4:5的宽高比使角色看起来更自然
                
                // 根据面板尺寸调整图像大小，保持宽高比
                if (panelWidth > 0 && panelHeight > 0) {
                    // 计算最大可用尺寸，保持宽高比
                    double scale = Math.min((double)panelWidth / imageWidth, (double)panelHeight / imageHeight);
                    imageWidth = (int)(imageWidth * scale * 0.8);  // 0.8缩放因子给周围留出一些空间
                    imageHeight = (int)(imageHeight * scale * 0.8);
                }
                
                // 计算居中位置
                int x = (panelWidth - imageWidth) / 2;
                int y = (panelHeight - imageHeight) / 2;
                
                // 绘制图像，保持正确的宽高比
                g.drawImage(currentFrame, x, y, imageWidth, imageHeight, null);
            }
        } else {
            // 如果没有动画帧，绘制一个默认的背景
            g.setColor(Color.LIGHT_GRAY);
            g.fillRect(0, 0, panel.getWidth(), panel.getHeight());
            g.setColor(Color.BLACK);
            g.drawString("加载动画中...", panel.getWidth()/2 - 30, panel.getHeight()/2);
        }
    }

    // 创建默认图像
    private Image createDefaultImage(String emotion) {
        BufferedImage image = new BufferedImage(100, 100, BufferedImage.TYPE_INT_ARGB);
        Graphics2D g2d = image.createGraphics();
        
        // 设置背景为透明
        g2d.setComposite(AlphaComposite.Clear);
        g2d.fillRect(0, 0, 100, 100);
        g2d.setComposite(AlphaComposite.SrcOver);
        
        // 根据情绪设置不同的默认颜色和形状
        switch (emotion) {
            case "开心":
                g2d.setColor(Color.YELLOW);
                break;
            case "难过":
                g2d.setColor(Color.BLUE);
                break;
            case "生气":
                g2d.setColor(Color.RED);
                break;
            case "疑惑":
                g2d.setColor(Color.PINK);
                break;
            default:
                g2d.setColor(Color.GREEN);
                break;
        }
        
        // 绘制一个简单的圆形作为默认图像
        g2d.fillOval(20, 20, 60, 60);
        g2d.dispose();
        
        return image;
    }

    // 初始化计时器
    private void initTimers() {
        // 动画播放计时器
        animationTimer = new Timer(frameRate, e -> {
            if (!isHolding && currentFrames != null && !currentFrames.isEmpty()) {
                currentFrameIndex = (currentFrameIndex + 1) % currentFrames.size();
                
                // 刷新面板以显示新帧
                if (animationPanel != null) {
                    animationPanel.repaint();
                }

                // 检查是否可以切换情绪（在特定帧）
                if (currentFrameIndex == 0) {
                    canSwitchEmotion = true;
                    // 如果有目标情绪，开始过渡
                    if (targetEmotion != null) {
                        startTransition(targetEmotion);
                        targetEmotion = null;
                    }
                }
            }
        });

        // 动画保持计时器
        holdTimer = new Timer(holdTime, e -> {
            isHolding = false;
            holdTimer.stop();
            animationTimer.start();
        });

        // 启动动画计时器
        animationTimer.start();
    }

    // 播放指定情绪的动画
    public void playEmotion(String emotion) {
        // 检查情绪是否有效
        if (!emotionFrames.containsKey(emotion)) {
            emotion = "平常";
        }

        // 如果当前正在保持，立即切换
        if (isHolding) {
            isHolding = false;
            holdTimer.stop();
            currentEmotion = emotion;
            currentFrames = emotionFrames.get(emotion);
            currentFrameIndex = 0;
            canSwitchEmotion = true;
            return;
        }

        // 如果可以切换情绪，立即切换
        if (canSwitchEmotion) {
            currentEmotion = emotion;
            currentFrames = emotionFrames.get(emotion);
            currentFrameIndex = 0;
            canSwitchEmotion = false;
        } else {
            // 否则设置为目标情绪，等待过渡
            targetEmotion = emotion;
        }
    }

    // 开始情绪过渡
    private void startTransition(String emotion) {
        currentEmotion = emotion;
        currentFrames = emotionFrames.get(emotion);
        currentFrameIndex = 0;
        transitionCounter = 0;
    }

    // 播放一次动画并保持最终状态
    public void playOnceAndHold(String emotion) {
        playEmotion(emotion);
        // 等待动画播放一次
        Timer waitTimer = new Timer(currentFrames.size() * frameRate, e -> {
            isHolding = true;
            animationTimer.stop();
            // 确保当前帧是最后一帧
            if (currentFrames != null && !currentFrames.isEmpty()) {
                currentFrameIndex = currentFrames.size() - 1;
                // 刷新面板
                if (animationPanel != null) {
                    animationPanel.repaint();
                }
            }
            holdTimer.start();
        });
        waitTimer.setRepeats(false);
        waitTimer.start();
    }

    // 停止动画
    public void stop() {
        animationTimer.stop();
        holdTimer.stop();
    }

    // 开始动画
    public void start() {
        if (!animationTimer.isRunning() && !isHolding) {
            animationTimer.start();
        }
    }

    // 获取当前情绪
    public String getCurrentEmotion() {
        return currentEmotion;
    }

    // 设置帧率
    public void setFrameRate(int frameRate) {
        this.frameRate = frameRate;
        animationTimer.setDelay(frameRate);
    }

    // 设置保持时间
    public void setHoldTime(int holdTime) {
        this.holdTime = holdTime;
        holdTimer.setDelay(holdTime);
    }

    // 根据情绪获取对应的颜文字
    public static String getEmotionText(String emotion) {
        switch (emotion) {
            case "生气":
                return TextStore.ANGRY;
            case "难过":
                return TextStore.SAD;
            case "开心":
                return TextStore.HAPPY;
            case "疑惑":
                return TextStore.CONFUSED;
            default:
                return TextStore.NORMAL;
        }
    }

    /**
     * 添加动画帧到指定情绪
     * 用于SpriteSheetLoader将从精灵表提取的帧添加到Behave实例
     * @param emotion 情绪名称
     * @param frames 帧列表
     */
    public void addAnimationFrames(String emotion, List<Image> frames) {
        if (emotion != null && frames != null && !frames.isEmpty()) {
            emotionFrames.put(emotion, new ArrayList<>(frames));
            // 如果当前正在播放该情绪，立即更新当前帧列表
            if (currentEmotion != null && currentEmotion.equals(emotion)) {
                currentFrames = emotionFrames.get(emotion);
                currentFrameIndex = 0;
                // 刷新面板
                if (animationPanel != null) {
                    animationPanel.repaint();
                }
            }
        }
    }

    /**
     * 重载paintCurrentFrame方法，用于在指定位置绘制当前帧
     * 兼容SpriteSheetDemo中的调用
     */
    public void paintCurrentFrame(Graphics g, int centerX, int centerY) {
        if (currentFrames != null && !currentFrames.isEmpty()) {
            Image currentFrame = currentFrames.get(currentFrameIndex);
            if (currentFrame != null) {
                // 计算图像居中显示的位置
                int imageWidth = TextStore.DEFAULT_SIZE;
                int imageHeight = TextStore.DEFAULT_SIZE;
                
                int x = centerX - imageWidth / 2;
                int y = centerY - imageHeight / 2;
                
                // 绘制图像
                g.drawImage(currentFrame, x, y, imageWidth, imageHeight, null);
            }
        }
    }
}