package org.example.ai;

import java.awt.BorderLayout;
import java.awt.Dimension;
import java.awt.Font;
import java.awt.GridLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.File;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.List;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFileChooser;
import javax.swing.JFrame;
import javax.swing.JLabel;
import javax.swing.JOptionPane;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.JTextArea;
import javax.swing.JTextField;
import javax.swing.SwingConstants;
import javax.swing.SwingWorker;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseMotionAdapter;
import java.awt.Image;

import org.example.tools.TextStore;

public class Appearance {
    private JFrame frame;
    private JTextArea chatArea;
    private JTextField inputField;
    private JButton sendButton;
    private JButton clearButton;
    private JButton feedDataButton;
    private JButton clearCharacterButton;
    private JButton statsButton;
    private JPanel emotionPanel;
    private Brain brain;
    private Behave behave; // 动画控制器
    private int mouseX, mouseY; // 用于窗口拖动

    public void setBrain(Brain brain) {
        this.brain = brain;
    }

    public void createAndShowGUI() {
        frame = new JFrame("AI聊天助手");
        frame.setDefaultCloseOperation(JFrame.EXIT_ON_CLOSE);
        frame.setSize(800, 700);
        frame.setLayout(new BorderLayout());
        
        // 设置窗口为完全透明、无边框、始终置顶
        frame.setUndecorated(true); // 去除窗口装饰（标题栏、边框等）
        frame.setAlwaysOnTop(true); // 窗口始终置顶
        frame.setBackground(new java.awt.Color(0, 0, 0, 0)); // 设置窗口背景为完全透明
        
        // 启用窗口透明度支持（在某些系统上可能需要）
        if (java.awt.GraphicsEnvironment.isHeadless()) {
            System.err.println("当前环境不支持图形界面");
        } else if (java.awt.GraphicsEnvironment.getLocalGraphicsEnvironment().getDefaultScreenDevice().isWindowTranslucencySupported(java.awt.GraphicsDevice.WindowTranslucency.PERPIXEL_TRANSPARENT)) {
            // 设置窗口为完全透明
            frame.setOpacity(1.0f); // 设置窗口不透明度为完全不透明（我们会通过面板实现透明效果）
        }

        // 初始化Behave动画控制器
        behave = Behave.getInstance();
        
        // 加载所有情绪的精灵表动画
        try {
            org.example.animation.SpriteSheetLoader.loadAllEmotionsForBehave(behave);
            System.out.println("精灵表动画加载成功");
        } catch (Exception e) {
            e.printStackTrace();
            System.err.println("加载精灵表动画时出错: " + e.getMessage());
        }

        // 创建主内容面板，使用BorderLayout
        JPanel contentPanel = new JPanel(new BorderLayout());
        contentPanel.setOpaque(false);

        // 创建动画显示面板 - 透明背景
        emotionPanel = new JPanel() {
            @Override
            protected void paintComponent(Graphics g) {
                // 不调用super.paintComponent，实现透明背景
                behave.paintCurrentFrame(g, this);
            }
        };
        emotionPanel.setPreferredSize(new Dimension(400, 400));
        emotionPanel.setOpaque(false); // 设置面板为透明
        
        // 设置动画面板，用于动画刷新
        behave.setAnimationPanel(emotionPanel);

        // 创建字幕显示面板 - 透明背景
        final JTextArea subtitleArea = new JTextArea();
        subtitleArea.setEditable(false);
        subtitleArea.setLineWrap(true);
        subtitleArea.setWrapStyleWord(true);
        subtitleArea.setOpaque(false); // 透明背景
        subtitleArea.setFont(new Font("宋体", Font.PLAIN, 16));
        subtitleArea.setForeground(java.awt.Color.WHITE); // 白色文字
        
        // 创建半透明背景的字幕面板
        JPanel subtitlePanel = new JPanel(new BorderLayout()) {
            @Override
            protected void paintComponent(Graphics g) {
                // 不调用super.paintComponent，实现更透明的效果
                Graphics2D g2d = (Graphics2D) g.create();
                g2d.setColor(new java.awt.Color(0, 0, 0, 128)); // 半透明黑色背景
                g2d.fillRect(0, 0, getWidth(), getHeight());
                g2d.dispose();
            }
        };
        subtitlePanel.setPreferredSize(new Dimension(400, 100));
        subtitlePanel.add(subtitleArea, BorderLayout.CENTER);
        subtitlePanel.setOpaque(false); // 设置面板为透明

        // 创建上方输入和按钮面板（保持透明，但位置可能需要调整）
        JPanel topPanel = new JPanel(new BorderLayout()) {
            @Override
            protected void paintComponent(Graphics g) {
                // 不调用super.paintComponent，实现更透明的效果
                Graphics2D g2d = (Graphics2D) g.create();
                g2d.setColor(new java.awt.Color(0, 0, 0, 150)); // 半透明黑色背景，更适合透明窗口
                g2d.fillRect(0, 0, getWidth(), getHeight());
                g2d.dispose();
            }
        };
        inputField = new JTextField();
        sendButton = new JButton("发送");
        clearButton = new JButton("清空历史");
        feedDataButton = new JButton("投喂数据");
        clearCharacterButton = new JButton("清空性格");
        statsButton = new JButton("统计");

        JPanel buttonPanel = new JPanel(new GridLayout(1, 5));
        buttonPanel.setOpaque(false); // 透明背景
        buttonPanel.add(sendButton);
        buttonPanel.add(clearButton);
        buttonPanel.add(feedDataButton);
        buttonPanel.add(clearCharacterButton);
        buttonPanel.add(statsButton);

        topPanel.setOpaque(false); // 透明背景
        topPanel.add(inputField, BorderLayout.CENTER);
        topPanel.add(buttonPanel, BorderLayout.EAST);
        topPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));

        // 创建中心面板，包含动画和字幕
        JPanel centerPanel = new JPanel(new BorderLayout());
        centerPanel.setOpaque(false);
        centerPanel.add(emotionPanel, BorderLayout.CENTER);
        centerPanel.add(subtitlePanel, BorderLayout.SOUTH);

        // 添加所有面板到内容面板
        contentPanel.add(topPanel, BorderLayout.NORTH);
        contentPanel.add(centerPanel, BorderLayout.CENTER);

        // 设置主窗口为透明，不再需要设置深色背景
        frame.setContentPane(contentPanel);
        
        // 添加窗口拖动功能
        frame.addMouseListener(new MouseAdapter() {
            @Override
            public void mousePressed(MouseEvent e) {
                mouseX = e.getX();
                mouseY = e.getY();
            }
        });
        
        frame.addMouseMotionListener(new MouseMotionAdapter() {
            @Override
            public void mouseDragged(MouseEvent e) {
                frame.setLocation(e.getXOnScreen() - mouseX, e.getYOnScreen() - mouseY);
            }
        });

        // 覆盖SendButtonListener以更新字幕显示
        sendButton.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                String userInput = inputField.getText().trim();
                if (!userInput.isEmpty()) {
                    sendButton.setEnabled(false);
                    inputField.setEnabled(false);
                    feedDataButton.setEnabled(false);
                    clearCharacterButton.setEnabled(false);
                    statsButton.setEnabled(false);

                    SwingWorker<String, Void> worker = new SwingWorker<String, Void>() {
                        @Override
                        protected String doInBackground() throws Exception {
                            return brain.getAIResponse(userInput);
                        }

                        @Override
                        protected void done() {
                            try {
                                String response = get();
                                
                                // 分析响应中的情绪，首先尝试从标签中提取
                                String emotion = "平常";
                                
                                // 尝试从响应开头提取情绪标签，如 [生气]、[开心] 等
                                java.util.regex.Pattern pattern = java.util.regex.Pattern.compile("^\\[([^\\]]+)\\]\\s*");
                                java.util.regex.Matcher matcher = pattern.matcher(response.trim());
                                if (matcher.find()) {
                                    String extractedEmotion = matcher.group(1).trim();
                                    // 验证提取的情绪是否有效
                                    if (extractedEmotion.equals("生气") || extractedEmotion.equals("难过") || 
                                        extractedEmotion.equals("开心") || extractedEmotion.equals("疑惑") || 
                                        extractedEmotion.equals("平常")) {
                                        emotion = extractedEmotion;
                                    }
                                } else {
                                    // 如果没有找到情绪标签，回退到原来的关键词分析方法
                                    if (response.contains("生气") || response.contains("愤怒")) {
                                        emotion = "生气";
                                    } else if (response.contains("难过") || response.contains("伤心")) {
                                        emotion = "难过";
                                    } else if (response.contains("开心") || response.contains("高兴")) {
                                        emotion = "开心";
                                    } else if (response.contains("疑惑") || response.contains("困惑")) {
                                        emotion = "疑惑";
                                    }
                                }

                                // 使用Behave类播放对应情绪的动画
                                behave.playEmotion(emotion);

                                // 更新字幕显示
                                subtitleArea.setText(response);

                            } catch (Exception ex) {
                                subtitleArea.setText("错误: " + ex.getMessage());
                            } finally {
                                sendButton.setEnabled(true);
                                inputField.setEnabled(true);
                                feedDataButton.setEnabled(true);
                                clearCharacterButton.setEnabled(true);
                                statsButton.setEnabled(true);
                                inputField.setText("");
                            }
                        }
                    };
                    worker.execute();
                }
            }
        });
        
        // 添加其他按钮的监听器
        clearButton.addActionListener(new ClearButtonListener());
        feedDataButton.addActionListener(new FeedDataButtonListener());
        clearCharacterButton.addActionListener(new ClearCharacterButtonListener());
        statsButton.addActionListener(new StatsButtonListener());
        inputField.addActionListener(new ActionListener() {
            @Override
            public void actionPerformed(ActionEvent e) {
                sendButton.doClick();
            }
        });

        frame.setVisible(true);
    }

    // 新增统计按钮监听器
    private class StatsButtonListener implements ActionListener {
        @Override
        public void actionPerformed(ActionEvent e) {
            brain.printKnowledgeStats();
            JOptionPane.showMessageDialog(frame,
                    "知识库统计信息已输出到控制台，请查看IDE或命令行窗口");
        }
    }

    private class ClearCharacterButtonListener implements ActionListener {
        @Override
        public void actionPerformed(ActionEvent e) {
            int result = JOptionPane.showConfirmDialog(frame, "确定要清空AI的性格（向量知识库）吗？这将删除所有学习的知识！", "清空性格", JOptionPane.YES_NO_OPTION);

            if (result == JOptionPane.YES_OPTION) {
                // 调用Brain的方法清空向量知识库
                brain.clearVectorStore();
                JOptionPane.showMessageDialog(frame, "AI性格（向量知识库）已清空");
            }
        }
    }

    private class FeedDataButtonListener implements ActionListener {
        @Override
        public void actionPerformed(ActionEvent e) {
            JFileChooser fileChooser = new JFileChooser();
            fileChooser.setDialogTitle("选择文本文件投喂给AI");
            fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
            fileChooser.setAcceptAllFileFilterUsed(false);
            fileChooser.addChoosableFileFilter(new javax.swing.filechooser.FileNameExtensionFilter("文本文件", "txt"));

            int result = fileChooser.showOpenDialog(frame);
            if (result == JFileChooser.APPROVE_OPTION) {
                File selectedFile = fileChooser.getSelectedFile();
                brain.importKnowledgeFromFile(selectedFile);
                JOptionPane.showMessageDialog(frame, String.format(TextStore.SYSTEM_KNOWLEDGE_IMPORTED, selectedFile.getName()));
            }
        }
    }

    private class ClearButtonListener implements ActionListener {
        @Override
        public void actionPerformed(ActionEvent e) {
            int result = JOptionPane.showConfirmDialog(frame,
                    "确定要清空对话历史吗？", "清空历史",
                    JOptionPane.YES_NO_OPTION);

            if (result == JOptionPane.YES_OPTION) {
                brain.clearConversationHistory();
                behave.playEmotion("平常"); // 恢复到平常表情
                JOptionPane.showMessageDialog(frame, "对话历史已清空");
            }
        }
    }
}